import csv, os, sys, argparse

from bl.vl.kb import KnowledgeBase as KB
import bl.vl.utils.ome_utils as vlu
from bl.vl.utils import LOG_LEVELS, get_logger


def make_parser():
    parser = argparse.ArgumentParser(description='Retrieve all enrollments')
    parser.add_argument('--logfile', type=str, help='log file (default=stderr)')
    parser.add_argument('--loglevel', type=str, choices = LOG_LEVELS,
                        help='logger level', default='INFO')
    parser.add_argument('--host', type=str, help='omero hostname')
    parser.add_argument('--user', type=str, help='omero user')
    parser.add_argument('--passwd', type=str, help='omero password')
    parser.add_argument('--ofile', type=str, help='output file path',
                        required=True)
    return parser


def main(argv):
    parser = make_parser()
    args = parser.parse_args(argv)

    # This is a temporary hack!!!
    to_be_ignored = ['IMMUNOCHIP_DISCARDED', 'CASI_MS_CSM_TMP',
                     'CASI_MS_CSM_CODES']

    logger = get_logger('all_enrollments', level=args.loglevel,
                        filename=args.logfile)
        
    try:
        host = args.host or vlu.ome_host()
        user = args.user or vlu.ome_user()
        passwd = args.passwd or vlu.ome_passwd()
    except ValueError, ve:
        logger.critical(ve)
        sys.exit(ve)

    try:
        out_file_path = args.ofile
    except IndexError:
        logger.error('Mandatory field missing.')
        parser.print_help()
        sys.exit(2)

    # Create the KnowledgeBase object
    kb = KB(driver='omero')(host, user, passwd)

    # Retrieve all studies from omero
    studies = kb.get_objects(kb.Study)
    studies = [s for s in studies if s.label not in to_be_ignored]
    logger.info('Retrieved %d studies from database' % len(studies))

    csv_header = ['individual_uuid']
    enrolls_map = {}

     # For each study, retrieve all enrollments
    for s in studies:
        logger.info('Retrieving enrollments for study %s' % s.label)
        enrolls = kb.get_enrolled(s)
        logger.info('%s enrollments retrieved' % len(enrolls))
        if len(enrolls) > 0:
            logger.debug('Building lookup dictionary....')
            for e in enrolls:
                enrolls_map.setdefault(e.individual.omero_id, {})['individual_uuid'] = e.individual.id
                enrolls_map[e.individual.omero_id].setdefault('studies', {})
                enrolls_map[e.individual.omero_id]['studies'].setdefault(s.label,[])
                enrolls_map[e.individual.omero_id]['studies'][s.label].append(e.studyCode)
                label = "{0} #{1}".format(s.label,len(enrolls_map[e.individual.omero_id]['studies'][s.label]))
                enrolls_map[e.individual.omero_id][label] = e.studyCode
                if label not in csv_header:
                    csv_header.append(label) # Add study label to CSV header
        else:
            logger.debug('No enrollments found, skip study %s' % s.label)

    # Write to CSV file
    logger.debug('Writing CSV file %s' % out_file_path)
    with open(out_file_path, 'w') as f:
        writer = csv.DictWriter(f, csv_header,
                                delimiter='\t', quotechar='"',
                                restval = 'None')
        writer.writeheader()
        for k, v in enrolls_map.iteritems():
            v.pop("studies",{})
            writer.writerow(v)

if __name__ == '__main__':
    main(sys.argv[1:])
