import argparse, sys

from bl.vl.kb import KnowledgeBase as KB
from bl.vl.utils import get_logger, LOG_LEVELS
import bl.vl.utils.ome_utils as vlu


def make_parser():
    parser = argparse.ArgumentParser(description='remove datasamples connected to a specific sample of a flowcell')
    parser.add_argument('--logfile', type=str, help='log file (default=stderr)')
    parser.add_argument('--loglevel', type=str, choices=LOG_LEVELS,
                        help='logging level (default=INFO)', default='INFO')
    parser.add_argument('-H', '--host', type=str, help='omero server hostname')
    parser.add_argument('-U', '--user', type=str, help='omero user')
    parser.add_argument('-P', '--passwd', type=str, help='omero password')
    parser.add_argument('--flowcell-label', type=str, required=True,
                        help='flowcell label')
    parser.add_argument('--sample-label', type=str, required=True,
                        help='sample label')
    parser.add_argument('--dry-run', action='store_true',
                        help='run a simulation, don\'t delete anything')
    return parser


def get_flowcell_samples_map(flowcell, kb, logger):
    fc = kb.get_by_label(kb.FlowCell, flowcell)
    if not fc:
        logger.info('No flowcell with label %s', flowcell)
        sys.exit(0)
    logger.info('Loading data for flowcell %s', flowcell)
    dsamples = kb.dt.get_connected(fc, kb.SeqDataSample)
    dsamples_map = {}
    for ds in dsamples:
        if ds.sample:
            dsamples_map.setdefault(ds.sample.label, []).append(ds)
        else:
            dsamples_map.setdefault('NO_SAMPLE', []).append(ds)
    return dsamples_map


def print_report(dsamples_map, sample_label, kb, logger):
    dsamples = dsamples_map.get(sample_label)
    if not dsamples:
        logger.info('No sample with label %s is related to the flowcell', sample_label)
        sys.exit(0)
    for ds in dsamples:
        dobjs = kb.get_data_objects(ds)
        logger.info('## data sample: %s', ds.label)
        for dob in dobjs:
            logger.info('### data object: %s --- mimetype: %s', dob.path, dob.mimetype)


def delete(dsamples_map, sample_label, kb, logger):
    for ds in dsamples_map[sample_label]:
        # this is a hack specific for the automator workflow
        if not ds.label.startswith('stage1'):
            logger.info('Deleting data for %s', ds.label)
            dobjs = kb.get_data_objects(ds)
            for d in dobjs:
                kb.delete(d)
            a = ds.action
            kb.delete(ds)
            try:
                kb.delete(a)
            except:
                pass


def main(argv):
    parser = make_parser()
    args = parser.parse_args(argv)

    logger = get_logger('delete_flowcell_results', level=args.loglevel,
                        filename=args.logfile)

    try:
        host = args.host or vlu.ome_host()
        user = args.user or vlu.ome_user()
        passwd = args.passwd or vlu.ome_passwd()
    except ValueError, ve:
        logger.critical(ve)
        sys.exit(ve)

    kb = KB(driver='omero')(host, user, passwd)

    dsamples_map = get_flowcell_samples_map(args.flowcell_label, kb, logger)
    print_report(dsamples_map, args.sample_label, kb, logger)
    if not args.dry_run:
        delete(dsamples_map, args.sample_label, kb. logger)
        pass
    else:
        logger.debug('SIMULATION, exit now')
    logger.info('Job completed')


if __name__ == '__main__':
    main(sys.argv[1:])