"""
From a list of platewells retrieves the enrollments code of the connected individual
"""
import argparse
import csv
import sys

from bl.vl.kb import KnowledgeBase as KB
import bl.vl.utils.ome_utils as vlu
from bl.vl.utils import LOG_LEVELS, get_logger

def make_parser():
  parser = argparse.ArgumentParser(description='From platewells 2 enrollments code')
  parser.add_argument('--logfile', type=str, help='log file (default=stderr)')
  parser.add_argument('--loglevel', type=str, choices = LOG_LEVELS,
                      help='logging level', default='INFO')
  parser.add_argument('-H', '--host', type=str, help='omero hostname')
  parser.add_argument('-U', '--user', type=str, help='omero user')
  parser.add_argument('-P', '--passwd', type=str, help='omero password')
  parser.add_argument('-S', '--study', type=str, required=True,
                      help='a study used to retrieve individuals')
  parser.add_argument('--ifile', type=str, required=True,
                      help='list of platewells used to fetch data')
  parser.add_argument('--ofile', type=str, help='output file path',
                      required=True)
  return parser


def main(argv):
  parser = make_parser()
  args = parser.parse_args(argv)

  logger = get_logger('pws2enrolls', level=args.loglevel,
                      filename=args.logfile)
  try:
    host = args.host or vlu.ome_host()
    user = args.user or vlu.ome_user()
    passwd = args.passwd or vlu.ome_passwd()
  except ValueError, ve:
    logger.critical(ve)
    sys.exit(ve)

  kb = KB(driver='omero')(host, user, passwd)
  study = kb.get_study(args.study)
  enrolled_map = {e.individual.id:e  for e in kb.get_enrolled(study)}
  logger.info('Loaded {} enrolled individuals for study {}'.format(len(enrolled_map), study.label))
  plates = kb.get_objects(kb.TiterPlate)
  logger.info('Loaded {} plates'.format(len(plates)))
  pws_map = {':'.join([w.container.barcode, w.label]):w for w in kb.get_objects(kb.PlateWell) 
             if w.container.barcode}
  logger.info('Loaded {} platewells'.format(len(pws_map)))
  #records = []
  of=open(args.ofile, 'w')
  writer=csv.DictWriter(of, ['platewell', 'status', 'enrollment'], 
                        delimiter='\t', quotechar='"', restval='None')
  writer.writeheader()
  with open(args.ifile, 'r') as f:
    reader=csv.DictReader(f, delimiter='\t')
    logger.info('Searching individuals connected to the platewells')
    for r in reader:
      ind=kb.dt.get_connected(pws_map[r['platewell']], kb.Individual, 
                              kb.dt.DIRECTION_INCOMING)
      try:          
        record = {'platewell': r['platewell'],
                  'status': pws_map[r['platewell']].status.enum_label(),
                  'enrollment': ':'.join([study.label,enrolled_map[ind[0].id].studyCode])}
      except KeyError as e:
        logger.warning('not enrolled {}'.format(r['platewell']))
        record = {'platewell': r['platewell'],
                  'status': pws_map[r['platewell']].status.enum_label(),
                  'enrollment': ':'.join([study.label,'not_enrolled'])}
      writer.writerow(record)
  of.close()

          
if __name__ == "__main__":
  main(sys.argv[1:])
