#-------------------------
# Export Galaxy history data to a JSON compatible text file
#
# Galaxy host can be retrieved using the GALAXY_HOST environment
# variable
# Galaxy API key can be retrieved using the GALAXY_API_KEY
# environment variable 
#-------------------------

from blend.galaxy import GalaxyInstance

import argparse, sys, logging, os, json

LOG_FORMAT = '%(asctime)s|%(levelname)-8s|%(message)s'
LOG_DATEFMT = '%Y-%m-%d %H:%M:%S'
LOG_LEVELS = ['DEBUG', 'INFO', 'WARNING', 'ERROR', 'CRITICAL']

def make_parser():
    parser = argparse.ArgumentParser(description='Export Galaxy history data')
    parser.add_argument('--loglevel', type=str, choices=LOG_LEVELS,
                        help='logging level (default: INFO)', default='INFO')
    parser.add_argument('--logfile', type=str, help='log file (default=stderr)')
    parser.add_argument('--galaxy_host', type=str, help='galaxy host (with port)')
    parser.add_argument('--api_key', type=str, help='Galaxy API key')
    parser.add_argument('--history_id', type=str, required=True,
                        help='ID fo the history that will be exported')
    parser.add_argument('--no_peek', action='store_true',
                        help='remove preview from datasets details')
    parser.add_argument('--ofile', type=str, required=True,
                        help='output file')
    return parser

def check_history_id(galaxy_instance, history_id):
    hmap = dict((str(h['id']), str(h['name'])) 
                for h in galaxy_instance.histories.get_histories())
    return history_id in hmap

def main(argv):
    parser = make_parser()
    args = parser.parse_args(argv)

    log_level = getattr(logging, args.loglevel)
    kwargs = {'format'  : LOG_FORMAT,
              'datefmt' : LOG_DATEFMT,
              'level'   : log_level}
    if args.logfile:
        kwargs['filename'] = args.logfile
    logging.basicConfig(**kwargs)
    logger = logging.getLogger('history_to_file')

    try:
        galaxy_host = args.galaxy_host or os.environ['GALAXY_HOST']
        api_key = args.api_key or os.environ['GALAXY_API_KEY']
    except KeyError, ke:
        msg = 'No argument passed and no global variable %s found' % ke
        logger.critical(msg)
        sys.exit(msg)

    logger.info('opening connection to %s' % galaxy_host)
    gi = GalaxyInstance(galaxy_host, key=api_key)

    if not check_history_id(gi, args.history_id):
        msg = 'Unable to find history with ID %s' % args.history_id
        logger.critical(msg)
        sys.exit(msg)

    logger.info('getting data for history %s' % args.history_id)
    h_data = gi.histories.get_history_details(args.history_id, args.no_peek)

    logger.info('saving to file %s' % args.ofile)
    with open(args.ofile, 'w') as out_file:
        out_file.write(json.dumps(h_data))
    logger.info('Job completed')

if __name__ == '__main__':
    main(sys.argv[1:])
