import argparse, sys
from collections import Counter

from bl.vl.kb import KnowledgeBase as KB
import bl.vl.utils.ome_utils as vlu
from bl.vl.utils import get_logger, LOG_LEVELS
from bl.vl.kb.drivers.omero.sequencing import SeqDataSample, SequencerOutput


def make_parser():
    parser = argparse.ArgumentParser(description='delete all items related to the given flowcell')
    parser.add_argument('--logfile', type=str, help='log file (default=stderr)')
    parser.add_argument('--loglevel', type=str, choices=LOG_LEVELS,
                        help='logging level (default=INFO)', default='INFO')
    parser.add_argument('-H', '--host', type=str, help='omero hostname')
    parser.add_argument('-U', '--user', type=str, help='omero user')
    parser.add_argument('-P', '--passwd', type=str, help='omero password')
    parser.add_argument('--flowcell-label', type=str, required=True,
                        help='flowcell label')
    parser.add_argument('--datasets-only', action='store_true',
                        help='delete only data samples and data objects related to the given flowcell')
    return parser


def get_sources(objs):
    sources = set()
    for x in objs:
        try:
            sources.add(x.action.target)
        except AttributeError:
            # Action has no "target" attribute, no source for item x
            pass
    return sources


def delete_objects(objs, kb, logger, max_retries = 3):
    retry_ct = Counter()
    while len(objs) > 0:
        o = objs.pop(0)
        if type(o) in [SeqDataSample, SequencerOutput]:
            logger.info('Loading DataObjects for %s:%s' % (o.__class__.__name__,
                                                           o.label))
            dobjs = kb.get_data_objects(o)
            logger.info('%d DataObjects loaded' % len(dobjs))
            for d in dobjs:
                logger.info('Deleting %s:%s' % (d.__class__.__name__,
                                                d.path))
                kb.delete(d)
        try:
            logger.info('Deleting %s:%s' % (o.__class__.__name__,
                                            o.id))
            act = o.action
            kb.delete(o)
            try:
                logger.info('Deleting source action %s:%s' % (act.__class__.__name__,
                                                              act.id))
                kb.delete(act)
            except:
                logger.info('Can\'t delete action')
        except:
            logger.info('Can\'t delete, putting back into objects list')
            if retry_ct['%s:%s' % (type(o), o.id)] < max_retries:
                objs.append(o)
                retry_ct['%s:%s' % (type(o), o.id)] += 1
            else:
                logger.info('Reached maximum retry limit for the object, skipping')

def main(argv):
    parser = make_parser()
    args = parser.parse_args(argv)

    logger = get_logger('drop_flowcell_related_items', level=args.loglevel,
                        filename=args.logfile)

    try:
        host = args.host or vlu.ome_host()
        user = args.user or vlu.ome_user()
        passwd = args.passwd or vlu.ome_passwd()
    except ValueError, ve:
        logger.critical(ve)
        sys.exit(ve)

    kb = KB(driver='omero')(host, user, passwd)

    logger.info('Retrieving flowcell with label %s' % args.flowcell_label)
    query = 'SELECT fc FROM FlowCell fc WHERE fc.label = :fc_label'
    results = kb.find_all_by_query(query, {'fc_label' : args.flowcell_label})
    if len(results) == 0:
        logger.info('No FlowCell with label %s in the database' % args.flowcell_label)
        sys.exit(0)
    fc = results[0]
    logger.info('Loaded FlowCell with ID %s' % fc.id)
    logger.info('Loading related Lanes')
    lanes = list(kb.get_lanes_by_flowcell(fc))
    logger.info('%d Lanes loaded' % len(lanes))
    logger.info('Loading related LaneSlots')
    lane_slots = []
    for l in lanes:
        lane_slots += list(kb.get_laneslots_by_lane(l))
    logger.info('%d LaneSlots loaded' % len(lane_slots))
    logger.info('Loading related Tubes')
    sub_samples = get_sources(lane_slots)
    samples = get_sources(sub_samples)
    logger.info('%d Tubes loaded' % (len(sub_samples) + len(samples)))
    logger.info('Loading related SequencerOutputs')
    seq_out = kb.dt.get_connected(fc, aklass = kb.SequencerOutput, direction = kb.dt.DIRECTION_OUTGOING)
    logger.info('%d SequencerOutputs loaded' % len(seq_out))
    logger.info('Loading related SeqDataSamples')
    seq_dsamples = kb.dt.get_connected(fc, aklass = kb.SeqDataSample, direction = kb.dt.DIRECTION_OUTGOING)
    logger.info('%d SeqDataSamples loaded' % len(seq_dsamples))

    if args.datasets_only:
        delete_items = [seq_dsamples]
    else:
        delete_items = [seq_dsamples, seq_out, lane_slots, lanes,
                        [fc], list(sub_samples), list(samples)]
    for items in delete_items:
        delete_objects(items, kb, logger)

    logger.info('Job completed')


if __name__ == '__main__':
    main(sys.argv[1:])
