# author: Pauline Ribeyre


#####################
# required packages #
#####################

library(batch) # provides "parseCommandargv" function


############
#   init   #
############

# read the parameters
argv_list <- parseCommandArgs(evaluate = FALSE) #, trailingOnly = TRUE)
argv <- unlist(argv_list)

# common environment to share variables between files
env <- new.env()

# timestamp made unique by adding a random element
env$timestamp <- round(as.numeric(Sys.time()) + runif(1, 0, 100))

# set working directory
tmpwd <- paste0("/galaxy/data/", env$timestamp)
# system("chmod -R 0777 .")
# dir.create(tmpwd, mode = "0777")
dir.create(tmpwd)
setwd(tmpwd)

# readme file
readme_msg <- paste0("\"", env$timestamp, "\"",
                     " is a temporary folder storing data for an instance of Galaxy tool \"GCMS-test\" launched on ",
                    Sys.time(), ". ",
                    "This folder should be deleted automatically at the end of the tool's execution. ",
                    "If the instance ended or has been aborted and this folder still exists, it can be deleted manually.")
write(readme_msg, file = "README.txt")

################
# source files #
################

source_local <- function(fname){
  # Gets a file's full name in Galaxy's directories.
  #
  # Args:
  #   fname: the name of the file.
  #
  # Returns:
  #   The full name of the file.

  argv <- commandArgs(trailingOnly = FALSE)
  base_dir <- dirname(substring(argv[grep("--file=", argv)], 8))
  return (paste(base_dir, fname, sep="/"))

}

env$source_settings <- source_local("GCMS-test_settings.R")
env$source_analyze <- source_local("GCMS-test_analyze.R")
env$source_output <- source_local("GCMS-test_output.R")
env$source_spectrum <- source_local("GCMS-test_spectrum.R")

env$source_file_names <- "file_names.txt"


#############
# functions #
#############

args_to_vary_list <- function() {
  # Creates the list of settings variations using the arguments.
  #
  # Returns:
  #   The list of settings variations.

  param <- c("step", "steps", "mzdiff", "fwhm", "simthresh", "snthresh", "max",
             "minclassfraction", "minclasssize", "rtdiff") #, "minfeat")

  PeakPicking_param <- c("step", "steps", "mzdiff", "fwhm", "snthresh", "max")
  betweenSamples_param <- c("min.class.fraction", "min.class.size", "rtdiff", "simthresh")

  vary <- list()

  for (p in param) {

    min <- as.numeric(argv[[paste0(p, "_min")]])
    max <- as.numeric(argv[[paste0(p, "_max")]])
    step <- as.numeric(argv[[paste0(p, "_step")]])

    if (p == "minclassfraction")
      p <- "min.class.fraction"
    else if (p == "minclasssize")
      p <- "min.class.size"

    if (p %in% PeakPicking_param)
      name <- paste0("PeakPicking$", p)
    else if (p %in% betweenSamples_param)
      name <- paste0("betweenSamples.", p)
    else
      stop("\"", p, "\" is not a valid parameter to vary.")

    # prevent seq error
    if (min > max)
      max <- min

    range <- seq(min, max, step)
    vary[[length(vary) + 1]] <- c(p, name, range)

  }

  return (vary)

}


read_ions_to_check <- function() {
  # Reads the arguments concerning the ions to check.
  
  env$ions_name <- list()
  env$ions_rt <- list()
  env$ions_mzs <- list()
  
  env$nb_ions_to_check <- length(grep("ions_name_", names(argv)))
  
  if (env$nb_ions_to_check > 0) {
    for (i in 1:env$nb_ions_to_check) {
      env$ions_name[length(env$ions_name) + 1] <- argv[[paste0("ions_name_", i)]]
      env$ions_rt[length(env$ions_rt) + 1] <- as.numeric(argv[[paste0("ions_rt_", i)]])
      mzs <- argv[[paste0("ions_mzs_", i)]]
      mzs <- as.numeric(strsplit(mzs, ",")[[1]])
      env$ions_mzs[length(env$ions_mzs) + 1] <- list(mzs)
    }
  }
  
}


############
#  inputs  #
############

env$nb_cores <- as.numeric(argv[["nb_cores"]])
cat("Nb cores:", env$nb_cores, "\n")

# get data from zip file
if (!is.null(argv[["zip_file"]])) {
  env$cdf_files <- unzip(argv[["zip_file"]])
} else {
  stop("No files have been provided.")
}

# vary parameters
env$vary_list <- args_to_vary_list()

# count duplicate ions
env$count_duplicates <- argv[["count_duplicates"]] == "true"
if (env$count_duplicates) {
  
  # delta rt
  env$duplicates_delta_rt <- as.numeric(argv[["duplicates_delta_rt"]])
  
  # delta mz
  env$duplicates_delta_mz <- as.numeric(argv[["duplicates_delta_mz"]])
  
}

# check the presence of ions
env$nb_ions_to_check <- 0
env$check_ions <- argv[["check_ions"]] == "true"
if (env$check_ions) {
  
  read_ions_to_check()
  
  # delta rt
  env$ions_delta_rt <- as.numeric(argv[["ions_delta_rt"]])
  
  # delta mz
  env$ions_delta_mz <- as.numeric(argv[["ions_delta_mz"]])
  
}


#############
#  outputs  #
#############

env$summary_out <- argv[["summary_out"]]
env$peakspectra_out <- argv[["peakspectra_out"]]
env$intensity_graph_out <- argv[["intensity_graph_out"]]

if (env$count_duplicates)
  env$count_duplicates_out <- argv[["duplicates_out"]]

env$compare_graph_out <- argv[["compare_graph_out"]]


##########
#  run   #
##########

cat("---------- Settings ----------\n")
sys.source(env$source_settings, env)


cat("---------- Analysis ----------\n")
sys.source(env$source_analyze, env)


cat("----------  Output  ----------\n")
sys.source(env$source_output, env)


##########
#  end   #
##########

# delete temporary files
setwd("..")
unlink(tmpwd, recursive = TRUE)
