# author: Pauline Ribeyre


#####################
# required packages #
#####################

library("metaMS") # provides "metaMSsettings" class


####################
# default settings #
####################

# PeakPicking
PeakPicking.method = "matchedFilter"
PeakPicking.step = 0.05
PeakPicking.steps = 2
PeakPicking.mzdiff = 0.5
PeakPicking.fwhm = 5
PeakPicking.snthresh = 2
PeakPicking.max = 500

# Alignment
# ?

# CAMERA
CAMERA.perfwhm = 1

# DBconstruction
DBconstruction.minintens = 0.0
DBconstruction.rttol = 0.1
DBconstruction.intensityMeasure = "maxo"
DBconstruction.DBthreshold = .80
DBconstruction.minfeat = 5

# match2DB
match2DB.rtdiff = 0.5
match2DB.minfeat = 5
# match2DB.rtval = ?
# match2DB.mzdiff = ?
# match2DB.ppm = ?
match2DB.simthresh = 0.8
match2DB.timeComparison = "rt"
match2DB.RIdiff = 5

# betweenSamples
betweenSamples.min.class.fraction = 0.8 # pools
# betweenSamples.min.class.fraction = 0.5 # samples
betweenSamples.min.class.size = 5
betweenSamples.timeComparison = "rt"
betweenSamples.rtdiff = 0.5
betweenSamples.RIdiff = 2 # default setting
# betweenSamples.RIdiff = 0.05 # w4m setting
betweenSamples.simthresh = 0.8

# matchIrrelevants
# matchIrrelevants.irrelevantClasses = c("Bleeding", "Plasticizers")
# matchIrrelevants.RIdiff = ?
# matchIrrelevants.rtdiff = ?
# matchIrrelevants.simthresh = ?
# matchIrrelevants.timeComparison = "RI"

# TODO RI option


###################
# settings object #
###################

# settings <- metaMSsettings("protocolName" = "settings", "chrom" = "GC")

settings <- metaMSsettings("protocolName" = "settings",
                            "chrom" = "GC",
                           
                            PeakPicking = list(
                              method = PeakPicking.method,
                              step = PeakPicking.step,
                              steps = PeakPicking.steps,
                              mzdiff = PeakPicking.mzdiff,
                              fwhm = PeakPicking.fwhm,
                              snthresh = PeakPicking.snthresh,
                              max = PeakPicking.max),
                           
                            CAMERA = list(perfwhm = CAMERA.perfwhm))

metaSetting(settings, "DBconstruction") <- list(
  minintens = DBconstruction.minintens,
  rttol = DBconstruction.rttol,
  intensityMeasure = DBconstruction.intensityMeasure,
  DBthreshold = DBconstruction.DBthreshold, 
  minfeat = DBconstruction.minfeat)

metaSetting(settings, "match2DB") <- list(
  simthresh = match2DB.simthresh,
  timeComparison = match2DB.timeComparison,
  rtdiff = match2DB.rtdiff,
  RIdiff = match2DB.RIdiff,
  minfeat = match2DB.minfeat)

metaSetting(settings, "betweenSamples") <- list(
  min.class.fraction = betweenSamples.min.class.fraction,
  min.class.size = betweenSamples.min.class.size,
  timeComparison = betweenSamples.timeComparison,
  rtdiff = betweenSamples.rtdiff,
  RIdiff = betweenSamples.RIdiff,    
  simthresh = betweenSamples.simthresh)

# metaSetting(GALAXY.GC, "matchIrrelevants") <- list(
#   irrelevantClasses = matchIrrelevants.irrelevantClasses,
#   RIdiff = matchIrrelevants.RIdiff,
#   rtdiff = matchIrrelevants.rtdiff,
#   simthresh = matchIrrelevants.simthresh,
#   timeComparison = matchIrrelevants.timeComparison)


# best settings pool
settings@betweenSamples.min.class.size = 5
settings@PeakPicking$fwhm = 6
settings@PeakPicking$mzdiff = -0.5
settings@betweenSamples.min.class.fraction = 0.8
settings@betweenSamples.simthresh = 0.7
settings@betweenSamples.rtdiff = 0.05


# best settings samples
settings@betweenSamples.min.class.size = 5
settings@PeakPicking$fwhm = 6
settings@PeakPicking$mzdiff = 0.5
settings@betweenSamples.min.class.fraction = 0.5
settings@betweenSamples.simthresh = 0.7
settings@betweenSamples.rtdiff = 0.05


#################
# vary settings #
#################

pop <- function(list) {
  # Removes the last item from a list.
  #
  # Args:
  #   list: the list to pop.
  #
  # Returns:
  #   The last item of the list before it was removed.
  
  val <- list[length(list)]
  assign(as.character(substitute(list)), list[-length(list)], parent.frame())
  return (val)
  
}


vary_list_to_settings <- function(settings, vary_list, title = "", titles_to_test = c(), settings_to_test = c()) {
  # Reads the list of parameters to vary and their ranges and builds a list of file titles and settings from it.
  #
  # Args:
  #   settings: default settings for runGC.
  #   vary_list: list of parameters to vary and the values each parameter must take.
  #   title: title of the current file (concatenation of the values taken by the varied parameters).
  #   titles_to_test:  list of titles (one for each settings set) (concatenation of the values taken by the varied parameters).
  #   settings_to_test: list of settings sets for runGC.
  #
  # Returns:
  #   A list containing:
  #   1: the list of file titles ;
  #   2: the list of runGC() settings.
  
  if (length(vary_list) == 0) { # if there are no parameters to vary
    
    titles_to_test <- c(titles_to_test, "default_settings")
    settings_to_test <- c(settings_to_test, settings)
    
  }
  
  else {
    
    param <- pop(vary_list)[[1]]
    
    title <- paste0(title, param[1])
    setting <- param[2]
    range <- as.numeric(param[3:length(param)])
    
    # for each possible value of the parameter
    for (p in range) {
      
      # change the settings
      isSlot <- regexpr("\\$", setting)[1]
      
      if (isSlot == -1) { # the parameter is a slot in class metaMSsettings
        slot(settings, setting) <- p
      }
      else { # the parameter is not a slot in class metaMSsettings
        slot(settings, substr(setting, 1, isSlot - 1))[substr(setting, isSlot + 1, nchar(setting))] <- p
      }
      
      new_title <- paste0(title, "=", p, "_")
      
      # save the settings in a list
      if (length(vary_list) == 0) {
        titles_to_test <- c(titles_to_test, new_title)
        settings_to_test <- c(settings_to_test, settings)
      }
      else {
        res <- vary_list_to_settings(settings, vary_list, new_title, titles_to_test, settings_to_test)
        titles_to_test <- res[[1]]
        settings_to_test <- res[[2]]
      }
      
    } # end for
    
  } # end else
  
  return (list(titles_to_test, settings_to_test))
  
}


############
#   main   #
############

# settings to vary and ranges of values
res <- vary_list_to_settings(settings, vary_list)
titles_to_test <- res[[1]]
settings_to_test <- res[[2]]
