import array
import math
import csv

def union(*input):
    o = {}
    for i in input:
        for a in i:
            o[a]= True
    return o.keys()

def intersect(a, b):
    o = []
    for i in a:
        if i in b:
            o.append(i)
    return o

class DataException(Exception):
    pass

class NamedVector(dict):
    
    def init_blank(self, names):
        for a in names:
            self[a] = float("nan")
    
    def __getitem__(self, key):
        if isinstance(key, NamedVector):
            o = NamedVector()
            for a in key:
                if key[a]:
                    o[a] = self.get(a)
            return o
        else:
            return self.get(key)
    
    
    def __add__(self, other):
        o = NamedVector()
        if isinstance(other, NamedVector):
            for a in union(self, other):
                o[a] = self.get(a, 0.0) + other.get(a, 0.0)
        else:
            for a in self:
                o[a] = self[a] + other
        return o
    
    def __div__(self, other):
        o = NamedVector()        
        if isinstance(other, NamedVector):
            for a in intersect(self, other):
                o[a] = self.get(a) / other.get(a)
        else:
            for a in self:
                o[a] = self[a] / other
        return o

    
    def __mul__(self, other):
        o = NamedVector()
        if isinstance(other, NamedVector):
            for a in intersect(self, other):
                o[a] = self.get(a) * other.get(a)
        else:
            for a in self:
                o[a] = self[a] * other
        return o
    
    def __sub__(self, other):
        o = NamedVector()
        if isinstance(other, NamedVector):
            for a in intersect(self, other):
                o[a] = self.get(a) - other.get(a)
        else:
            for a in self:
                o[a] = self[a] - other
        return o
    
    def __gt__(self, other):
        o = NamedVector()
        if isinstance(other, NamedVector):
            for a in intersect(self, other):
                o[a] = self.get(a) > other.get(a)
        else:
            for a in self:
                o[a] = self[a] > other
        return o

    def __eq__(self, other):
        o = NamedVector()
        if isinstance(other, NamedVector):
            for a in intersect(self, other):
                o[a] = self.get(a) == other.get(a)
        else:
            for a in self:
                o[a] = self[a] == other
        return o

    def __ne__(self, other):
        o = NamedVector()
        if isinstance(other, NamedVector):
            for a in intersect(self, other):
                o[a] = self.get(a) != other.get(a)
        else:
            for a in self:
                o[a] = self[a] != other
        return o


    def __lt__(self, other):
        o = NamedVector()
        if isinstance(other, NamedVector):
            for a in intersect(self, other):
                o[a] = self.get(a) < other.get(a)
        else:
            for a in self:
                o[a] = self[a] < other
        return o

    def pow(self, other):
        o = NamedVector()
        if isinstance(other, NamedVector):
            for a in intersect(self, other):
                o[a] = math.pow(self.get(a), other.get(a))
        else:
            for a in self:
                o[a] = math.pow(self[a], other)
        return o

    def log(self, base=math.e):
        o = NamedVector()
        for a in self:
            o[a] = math.log(self[a], base)
        return o

    def sqrt(self):
        o = NamedVector()
        for a in self:
            o[a] = math.sqrt(self[a])
        return o
        
    def array(self, names):
        o = array.array('f')
        for a in names:
            o.append( self.get(a, float('nan')) )
        return o

    def set_nan(self, value=0.0):
        for a in self:
            if math.isnan(self[a]):
                self[a] = value
    
    def sum(self):
        return sum(self.values())
        
    

class FloatMatrix:
    """
    array.array based float matrix class
    """
    def __init__(self):
        self.corner_name = "probe"
        self.data = None
        self.nrows = None
        self.ncols = None
        self.rowmap = None
        self.colmap = None

    def read(self, handle):
        header = None
        for line in handle:
            row = line.rstrip().split("\t")
            if header is None:
                header = row
                self.data = array.array("f")
                self.colmap = {}
                self.rowmap = {}
                self.ncols = len(row) - 1
                self.nrows = 0
                for i, c in enumerate(row[1:]):
                    self.colmap[c] = i
            else:
                if len(row) - 1 != self.ncols:
                    raise DataException("Misformed matrix")
                self.rowmap[row[0]] = len(self.rowmap)
                a = []
                for v in row[1:]:
                    try:
                        a.append(float(v))
                    except ValueError:
                        a.append(float('Nan'))
                self.data.extend(a)
                self.nrows += 1

    def init_blank(self, rows, cols):
        self.data = array.array("f")
        self.colmap = {}
        for i,c in enumerate(cols):
            self.colmap[c] = i
        self.rowmap = {}
        for i,r in enumerate(rows):
            self.rowmap[r] = i
        self.ncols = len(cols)
        self.nrows = len(rows)
        for i in range(self.nrows):
            self.data.extend([float('nan')] * self.ncols)


    def size(self):
        return {'rows' : len(self.rowmap), 'cols' : len(self.colmap)}

    def has_row(self, row_name):
        return row_name in self.rowmap

    def has_col(self, col_name):
        return col_name in self.colmap

    def values(self):
        return self.data

    def get_value(self, row_name, col_name):
        return self.data[ self.rowmap[row_name] * self.ncols + self.colmap[col_name] ]

    def set_value(self, row_name, col_name, value):
        self.data[ self.rowmap[row_name] * self.ncols + self.colmap[col_name] ] = value
    
    def get_row(self, row_name):
        if row_name not in self.rowmap:
            raise KeyError
        out = NamedVector()
        out.init_blank( self.colmap )
        for c in self.colmap:
            out[c] = self.data[ self.rowmap[row_name] * self.ncols + self.colmap[c] ]
        return out

    def get_col(self, col_name):
        if col_name not in self.colmap:
            raise KeyError
        out = NamedVector()
        out.init_blank( self.rowmap )
        for r in self.rowmap:
            out[r] = self.data[ self.rowmap[r] * self.ncols + self.colmap[col_name] ]
        return out


    def set_row(self, row_name, row_data):
        if row_name not in self.rowmap:
            raise KeyError
        row_offset = self.rowmap[row_name] * self.ncols
        for c in row_data:
            if c in self.colmap:
                self.data[ row_offset + self.colmap[c] ] = row_data[c] 

    def get_cols(self):
        if self.colmap is None:
            return None
        return self.colmap.keys()

    def get_rows(self):
        if self.rowmap is None:
            return None
        return self.rowmap.keys()

    def select(self, rows=None, cols=None):
        if rows is None:
            rows = self.get_rows()
        if cols is None:
            cols = self.get_cols()
        out = FloatMatrix()
        out.init_blank(rows=rows, cols=cols)
        for row in rows:
            if self.has_row(row):
                r = self.get_row(row)
                out.set_row(row, r)
        return out

    
    def write(self, handle, missing='NA', row_select=None, col_select=None):
        write = csv.writer(handle, delimiter="\t", lineterminator='\n')

        col_list = []
        if col_select is None:
            col_list = self.get_cols()
        else:
            for c in self.get_cols():
                if c in col_select:
                    col_list.append(c)
        
        write.writerow([self.corner_name] + col_list)
        for rowName in self.rowmap:
            if row_select is None or rowName in row_select:
                out = [rowName]
                row = self.get_row(rowName)
                for col in col_list:
                    val = row[col]
                    if val is None or math.isnan(val):
                        val = missing
                    else:
                        val = "%.5f" % (val)
                    out.append(val)
                write.writerow(out)
    
    def set_nan(self, value=0.0):
        for i in range(len(self.data)):
            if math.isnan(self.data[i]):
                self.data[i] = value
    
    def row_sums(self):
        out = NamedVector()
        for rowName in self.rowmap:
            row = self.get_row(rowName)
            out[rowName] = row.sum()
        return out
    
    def merge(self, other):
        out = FloatMatrix()
        out.init_blank( rows=union(self.get_rows(), other.get_rows()), cols=union( self.get_cols(), other.get_cols()))
        
        for rowName in self.rowmap:
            for colName in self.colmap:
                out.set_value( row_name=rowName, col_name=colName, value=self.get_value(row_name=rowName, col_name=colName))

        for rowName in other.rowmap:
            for colName in other.colmap:
                out.set_value( row_name=rowName, col_name=colName, value=other.get_value(row_name=rowName, col_name=colName))
        
        return out

    def toRmatrix(self, r):
        out = r.matrix(self.data, ncol=self.ncols, dimnames=[ self.get_rows(), self.get_cols() ], byrow=True)
        return out


