#!/usr/bin/env python

import sys
import time
import array
import csv
import math
import ctypes
import ctypes.util
import argparse

libc = ctypes.cdll.LoadLibrary(ctypes.util.find_library("c"))

def rankTransform(rank, total):
    return rank / float(total)

def py_cmp_float(a_ptr, b_ptr):
    a = a_ptr.contents.value
    b = b_ptr.contents.value
    if a < b:
        return -1
    return a > b

CMPFUNC = ctypes.CFUNCTYPE(ctypes.c_int, ctypes.POINTER(ctypes.c_float), ctypes.POINTER(ctypes.c_float))

cmp_float = CMPFUNC(py_cmp_float)

def csort(buf):
    """
    This is an inplace sort of an array.array float class using the C qsort function
    """
    addr, count = buf.buffer_info()
    libc.qsort( ctypes.cast(addr, ctypes.POINTER(ctypes.c_float)), count, ctypes.sizeof(ctypes.c_float), cmp_float)     


def transformFile(fh, ofh, sep="\t", zero_drop=False, na2zero=False):

    floatValues = None
    cols = None
    rows = None
    totalValues = 0
    reader = csv.reader(fh, delimiter=sep)
    for row in reader:
        if cols is None:
            cols = row[1:]
            numCols = len(cols)
            rows = []
            floatValues = array.array('f')
        else:
            rows.append(row[0])
            assert(len(row)-1 == numCols)
            for val in row[1:]:
                try:
                    v = float(val)
                    floatValues.append(v)
                    if not zero_drop or v != 0.0:
                        totalValues += 1
                except ValueError:
                    floatValues.append(float('nan'))

    numRows = len(rows)
    if (numRows == 0):
        sys.stderr.write("Empty input\n")
        exit(10)

    if totalValues == 0:
        assert False, "did not read any values"
    
    sortedValues = array.array('f')
    for f in floatValues:
        if f == f:
             sortedValues.append(f)
    
    csort(sortedValues)

    i = 0
    rankDict = dict()
    for val in sortedValues:
        if not math.isnan(val) and (not zero_drop or val != 0.0):
            rankDict[val] = rankTransform(i, totalValues)
            i += 1

    def rowString(rowNum):
        def matrixVal(colNum):
            val = floatValues[rowNum*numCols + colNum]
            if val in rankDict: 
                return ("%5g" % rankDict[val])
            else:
                if na2zero:
                    return "0"
                else:
                    return "NA"
        return "\t".join(map(matrixVal, range(numCols)))

    ofh.write( "probe\t%s\n" % ("\t".join(cols)) )
    for j in range(numRows):
        ofh.write("%s\t%s\n" % (rows[j], rowString(j)))

def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("-s", "--sep", help="Seperator", default="\t")
    parser.add_argument("-o", "--out", help="Output", default=None)    
    parser.add_argument("-n", "--na2zero", help="Change NAs to Zero", action="store_true", default=False)    
    parser.add_argument("-z", "--zero-drop", help="Drop Zero Values", action="store_true", default=False)
    parser.add_argument("input", help="Input Matrix", default=None)
    
    args = parser.parse_args()
    
    if args.input == "-":
        fh = sys.stdin
    else:
        fh = open(args.input)
    if args.out is None:
        ofh = sys.stdout
    else:
        ofh = open(args.out, "w")

    transformFile(fh, ofh, args.sep, args.zero_drop, args.na2zero)
    fh.close()
    ofh.close()

if __name__ == "__main__":
    main()
