#!/usr/bin/env python
#Dan Blankenberg

import sys
import os
import tempfile
import optparse
import subprocess

from galaxy.util.json import from_json_string, to_json_string

CHUNK_SIZE = 2**20

DEFAULT_DATA_TABLE_NAME = "twobit"

def build_twobit_index( data_manager_dict, fasta_filename, target_directory, dbkey, data_table_name=DEFAULT_DATA_TABLE_NAME ):
    #TODO: allow multiple FASTA input files
    fasta_base_name = os.path.split( fasta_filename )[-1]
    twobit_base_name = os.path.splitext(fasta_base_name)[0] + '.2bit'
    twobit_path =  os.path.join( target_directory,twobit_base_name )
    sym_linked_fasta_filename = os.path.join( target_directory, fasta_base_name )
    os.symlink( fasta_filename, sym_linked_fasta_filename )
    
    args = [ 'faToTwoBit' ]
    args.append( sym_linked_fasta_filename )
    args.append( twobit_path )
    tmp_stderr = tempfile.NamedTemporaryFile( prefix = "tmp-data-manager-twobit_builder-stderr" )
    proc = subprocess.Popen( args=args, shell=False, cwd=target_directory, stderr=tmp_stderr.fileno() )
    return_code = proc.wait()
    if return_code:
        tmp_stderr.flush()
        tmp_stderr.seek( 0 )
        sys.stderr.write( "Error building index:\n" )
        while True:
            chunk = tmp_stderr.read( CHUNK_SIZE )
            if not chunk:
                break
            sys.stderr.write( chunk )
        sys.exit( return_code )
    tmp_stderr.close()
    data_table_entry = dict( value=dbkey, path=twobit_base_name )
    _add_data_table_entry( data_manager_dict, data_table_name, data_table_entry )

def _add_data_table_entry( data_manager_dict, data_table_name, data_table_entry ):
    data_manager_dict['data_tables'] = data_manager_dict.get( 'data_tables', {} )
    data_manager_dict['data_tables'][ data_table_name ] = data_manager_dict['data_tables'].get( data_table_name, [] )
    data_manager_dict['data_tables'][ data_table_name ].append( data_table_entry )
    return data_manager_dict

def main():
    #Parse Command Line
    parser = optparse.OptionParser()
    parser.add_option( '-f', '--fasta_filename', dest='fasta_filename', action='store', type="string", default=None, help='fasta_filename' )
    parser.add_option( '-d', '--fasta_dbkey', dest='fasta_dbkey', action='store', type="string", default=None, help='fasta_dbkey' )
    parser.add_option( '-n', '--data_table_name', dest='data_table_name', action='store', type="string", default=None, help='data_table_name' )
    (options, args) = parser.parse_args()
    
    filename = args[0]
    
    params = from_json_string( open( filename ).read() )
    target_directory = params[ 'output_data' ][0]['extra_files_path']
    os.mkdir( target_directory )
    data_manager_dict = {}
    
    if options.fasta_dbkey in [ None, '', '?' ]:
        raise Exception( '"%s" is not a valid dbkey. You must specify a valid dbkey.' % ( dbkey ) )
    
    #build the index
    build_twobit_index( data_manager_dict, options.fasta_filename, target_directory, options.fasta_dbkey,  data_table_name=options.data_table_name or DEFAULT_DATA_TABLE_NAME )
    
    #save info to json file
    open( filename, 'wb' ).write( to_json_string( data_manager_dict ) )
        
if __name__ == "__main__": main()
