import argparse
import json
import os
import sys
try:
    # For Python 3.0 and later
    from urllib.request import Request, urlopen
except ImportError:
    # Fall back to Python 2 imports
    from urllib2 import Request, urlopen

BASE_URL = 'http://bigg.ucsd.edu/static/models'
ID2ORG = {
    "iCN718": "Acinetobacter baumannii AYE",
    "iYO844": "Bacillus subtilis subsp. subtilis str. 168",
    "iRC1080": "Chlamydomonas reinhardtii",
    "iCN900": "Clostridioides difficile 630",
    "iHN637": "Clostridium ljungdahlii DSM 13528",
    "iCHOv1_DG44": "Cricetulus griseus",
    "iCHOv1": "Cricetulus griseus",
    "iAF1260b": "Escherichia coli str. K-12 substr. MG1655",
    "iAF1260": "Escherichia coli str. K-12 substr. MG1655",
    "iML1515": "Escherichia coli str. K-12 substr. MG1655",
    "iJO1366": "Escherichia coli str. K-12 substr. MG1655",
    "iJR904": "Escherichia coli str. K-12 substr. MG1655",
    "e_coli_core": "Escherichia coli str. K-12 substr. MG1655",
    "iAF987": "Geobacter metallireducens GS-15",
    "iIT341": "Helicobacter pylori 26695",
    "iAT_PLT_636": "Homo sapiens",
    "Recon3D": "Homo sapiens",
    "iAB_RBC_283": "Homo sapiens",
    "RECON1": "Homo sapiens",
    "iYL1228": "Klebsiella pneumoniae subsp. pneumoniae MGH 78578",
    "iNF517": "Lactococcus lactis subsp. cremoris MG1363",
    "iAF692": "Methanosarcina barkeri str. Fusaro",
    "iMM1415": "Mus musculus",
    "iNJ661": "Mycobacterium tuberculosis H37Rv",
    "iEK1008": "Mycobacterium tuberculosis H37Rv",
    "iLB1027_lipid": "Phaeodactylum tricornutum CCAP 1055/1",
    "iAM_Pb448": "Plasmodium berghei",
    "iAM_Pc455": "Plasmodium cynomolgi strain B",
    "iAM_Pf480": "Plasmodium falciparum 3D7",
    "iAM_Pk459": "Plasmodium knowlesi strain H",
    "iAM_Pv461": "Plasmodium vivax Sal-1",
    "iJN746": "Pseudomonas putida KT2440",
    "iJN1463": "Pseudomonas putida KT2440",
    "iND750": "Saccharomyces cerevisiae S288C",
    "iMM904": "Saccharomyces cerevisiae S288C",
    "STM_v1_0": "Salmonella enterica subsp. enterica serovar Typhimurium str. LT2",
    "iYS1720": "Salmonella pan-reactome",
    "iSB619": "Staphylococcus aureus subsp. aureus N315",
    "iYS854": "Staphylococcus aureus subsp. aureus USA300_TCH1516",
    "iJB785": "Synechococcus elongatus PCC 7942",
    "iJN678": "Synechocystis sp. PCC 6803",
    "iSynCJ816": "Synechocystis sp. PCC 6803",
    "iLJ478": "Thermotoga maritima MSB8",
    "iIS312": "Trypanosoma cruzi Dm28c",
    "iIS312_Trypomastigote": "Trypanosoma cruzi Dm28c",
    "iIS312_Epimastigote": "Trypanosoma cruzi Dm28c",
    "iIS312_Amastigote": "Trypanosoma cruzi Dm28c"
}

def url_download(url, workdir):
    file_path = os.path.abspath(os.path.join(workdir, os.path.basename(url)))
    src = None
    dst = None
    try:
        req = Request(url)
        src = urlopen(req)
        with open(file_path, 'wb') as dst:
            while True:
                chunk = src.read(2**10)
                if chunk:
                    dst.write(chunk)
                else:
                    break
    except Exception as e:
        sys.exit(str(e))
    finally:
        if src:
            src.close()
    return file_path


def download(model_id, out_file):

    with open(out_file) as fh:
        params = json.load(fh)

    workdir = params['output_data'][0]['extra_files_path']
    os.makedirs(workdir)

    data_manager_json = {'data_tables': {}}
    file_path = url_download(BASE_URL + '/' + model_id + '.xml', workdir)

    data_manager_entry = {}
    data_manager_entry['value'] = model_id
    data_manager_entry['name'] = ID2ORG.get(model_id, 'undefined')
    data_manager_entry['path'] = file_path

    data_manager_json['data_tables']['bigg_model_sbml'] = data_manager_entry
    with open(out_file, 'w') as fh:
        json.dump(data_manager_json, fh, sort_keys=True)


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('--model-id', help='Model BIGG id')
    parser.add_argument('--out-file', help='JSON output file')
    args = parser.parse_args()

    download(args.model_id, args.out_file)
