#include "Alignment.h"
#include "int_util.h"

#include <boost/algorithm/string.hpp>

#include <iostream>
#include <sstream>
#include <algorithm>

#define READ_PAIRED                         1
#define READ_MAPPED_IN_PROPER_PAIR          2
#define READ_UNMAPPED                       4
#define MATE_UNMAPPED                       8
#define READ_REVERSE_STRAND                 16
#define MATE_REVERSE_STRAND                 32
#define FIRST_IN_PAIR                       64
#define SECOND_IN_PAIR                      128
#define NOT_PRIMARY_ALIGNMENT               256
#define READ_FAILS_VENDOR_QUALITY_CHECKS    512
#define READ_IS_PCR_OR_OPTICAL_DUPLICATE    1024
#define SUPPLEMENTARY_ALIGNMENT             2048

Alignment::Alignment(std::string alignment) : _alignment(alignment) {
	fill_alignment_fields(alignment);
	fill_bit_flag(field.FLAG);
	fill_xa_field(alignment);
}

void Alignment::fill_alignment_fields(const std::string &alignment) {
	std::istringstream ss(alignment);
	ss >> field.QNAME >> field.FLAG >> field.RNAME >> field.POS >>
              field.MAPQ >> field.CIGAR >> field.RNEXT >> field.PNEXT >>
              field.TLEN >> field.SEQ >> field.QUAL;
}

void Alignment::fill_bit_flag(const int &flag) {
    if( (flag & READ_PAIRED) == 1)                     b_flag.read_paired = true;
    if( (flag & READ_MAPPED_IN_PROPER_PAIR) > 1)       b_flag.read_mapped_in_proper_pair = true;
    if( (flag & READ_UNMAPPED) > 1)                    b_flag.read_unmapped = true;
    if( (flag & MATE_UNMAPPED) > 1)                    b_flag.mate_unmapped = true;
    if( (flag & READ_REVERSE_STRAND) > 1)              b_flag.read_reverse_strand = true;
    if( (flag & MATE_REVERSE_STRAND) > 1)              b_flag.mate_reverse_strand = true;
    if( (flag & FIRST_IN_PAIR) > 1)                    b_flag.first_in_pair = true;
    if( (flag & SECOND_IN_PAIR) > 1)                   b_flag.second_in_pair = true;
}

// XA:Z:mef(A)_10_AF376746,+81,92M,2;
bool Alignment::fill_xa_field(const std::string &alignment) {
	xa_fields hits;
	std::string alt_fields;
	int offset = 5;
	std::size_t alt_index = alignment.find("XA:Z:");
	
	if(alt_index != std::string::npos) {
		alt_fields = alignment.substr(alt_index+offset, alignment.length());		
	}
	else {
		return false;
	}

	std::string field;
	std::istringstream ss(alt_fields);

	while(std::getline(ss, field, ',')) {
		hits.rname = field;
		//std::cout << hits.rname << std::endl;
		getline(ss, field, ',');
		hits.pos = strand_to_i(field);
		//std::cout << hits.pos << std::endl;
		getline(ss, field, ',');
		hits.cigar = field;
		//std::cout << hits.cigar << std::endl;
		getline(ss, field, ';');
		hits.edit = 0;
		alternate_hits.push_back(hits);
	}
		
	return true;	
}

std::vector<std::pair<int,char> > Alignment::get_cigar() {
	return get_cigar_operations(field.CIGAR);
}

std::vector<std::pair<int,char> > Alignment::get_cigar_operations(const std::string &cigar) {
	std::vector<std::pair<int,char>> p;
	int count;
	char operation;
	
	std::istringstream ss(cigar);
	while(ss >> count >> operation) {
		p.push_back(std::make_pair(count,operation));
	}

	return p;
}
